<?php
/**
 * Plugin Name: TGEI Auto Tabs
 * Description: Converts a page or post into a series of tabs
 * Version: 1.0
 * Stable: 1.0
 * Author: Too Good Enterprises Incorporated
 * Author URI: https://toogoodenterprises.com
 * License: GPLv2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */
namespace TooGoodEnterprisesInc\AutoTabs;
use Dom;
defined("ABSPATH") || exit;

// import hte settings class
class TGEI_AutoTabs
{
  const ENABLED = "1";
  const DISABLED = "off";


  public static function register()
  {
    add_action("plugins_loaded", __NAMESPACE__."\TGEI_AutoTabs::initialize", 10);
    register_activation_hook(__FILE__, __NAMESPACE__."\TGEI_AutoTabs::activate");
    register_deactivation_hook(__FILE__, __NAMESPACE__."\TGEI_AutoTabs::deactivate");
    register_uninstall_hook(__FILE__, __NAMESPACE__."\TGEI_AutoTabs::uninstall");
  }

  public static function activate(){}
  public static function deactivate(){}
  public static function uninstall(){}

  public static function initialize()
  {
    // register a post meta that will keep track whether the post or page is allowed
    register_post_meta(
      "",
      "tgei-auto-tabs-enabled",
      array(
        "type" => "boolean",
        "default" => false,
        "single" => true,
      )
    );


    // filter the post to automatically put in the tabs
    add_filter('the_content', __NAMESPACE__."\TGEI_AutoTabs::addTabs", 10, 1);

    // actions and filters for admin section only
    if(is_admin())
    {
      // initialize the meta boxes in the post and page editor
      add_action("add_meta_boxes", __NAMESPACE__."\TGEI_AutoTabs::addMetaBox");
      add_action( "save_post", __NAMESPACE__."\TGEI_AutoTabs::saveStatusPost" );
    }
    else
    {
      add_action("wp_enqueue_scripts", __NAMESPACE__."\TGEI_AutoTabs::enqueueStyles");
    }
  }

  public static function enqueueStyles()
  {
    wp_enqueue_style("TGEI_AutoTabs_CSS", plugins_url("TGEI-AutoTabs/style.css"), [], wp_rand());
    wp_enqueue_script("TGEI_AutoTabs_JS", plugins_url("TGEI-AutoTabs/script.js"), [], false, wp_rand(), array("in_footer" => true));
  }
  
  public static function addMetaBox()
  {
    $allPostTypes = get_post_types();
    add_meta_box(
      "tgei-auto-tabs",
      "TGEI Auto Tabs",
      __NAMESPACE__."\TGEI_AutoTabs::postMetaBoxRender",
      $allPostTypes,
      "side"
    );
  }

  public static function postMetaBoxRender()
  {
    global $post;
    $checked = get_post_meta($post->ID, "tgei-auto-tabs-enabled", true) === TGEI_AutoTabs::ENABLED ? "checked" :"";
    $tgeiNonce = wp_create_nonce( "tgei-auto-tabs-nonce");
    echo '<p>Render page sections as tabs.</p>';
    echo '<input type="hidden" name="tgei-auto-tabs-enabled" value="off" />';
    echo '<label class="tgei-toggle">Enable: <input type="checkbox" name="tgei-auto-tabs-enabled" id="tgei-auto-tabs-enabled" '.esc_attr($checked).' /></label>';
    echo '<input type="hidden" name="tgei-auto-tabs-nonce" value="'.esc_attr($tgeiNonce).'" />';
  }

  public static function saveStatusPost($postId)
  {
    // check we have a tgei-semi-headless-allow field
    // if not, skip
    if(array_key_exists("tgei-auto-tabs-enabled", $_POST))
    {
      // nonce check
      if (!array_key_exists("tgei-auto-tabs-nonce", $_POST ) || ! wp_verify_nonce( sanitize_key(wp_unslash($_POST[ "tgei-auto-tabs-nonce"])), "tgei-auto-tabs-nonce" ) ) {
        return;
      }
      $enabled = TGEI_AutoTabs::DISABLED;
      // check if status set to on
      if($_POST["tgei-auto-tabs-enabled"] == "on")
      {
        $enabled = TGEI_AutoTabs::ENABLED;
      }
      update_post_meta($postId,"tgei-auto-tabs-enabled", $enabled);
    }

  }

  public static function addTabs($content)
  {
    if(!is_admin())
    {
      global $post;
      $addTabs = get_post_meta($post->ID, "tgei-auto-tabs-enabled", true) === TGEI_AutoTabs::ENABLED;
      if($addTabs === true)
      {
        return TGEI_AutoTabs::render($content);
      }
    }
    return $content;
  }

  public static function render($content)
  {
    $contentMod = "";
    $h1Index = strpos($content, "<h1");
    $contentMod = substr($content, 0, $h1Index);// copy all the html elements before the first h1 as is
    $h1CloseIndex = false;
    $tabTitles = array();
    $tabContent = array();
    while($h1Index !== false)
    {
      // all elements between the previous h1 and the current h1 are the content of the previous tab
      if(count($tabTitles) > 0)
      {
        array_push($tabContent, substr($content,$h1CloseIndex + 5, $h1Index - $h1CloseIndex - 5)); 
      }
      $endOfStartTag = strpos($content, ">", $h1Index);
      $h1CloseIndex = strpos($content, "</h1>", $h1Index);
      array_push($tabTitles, substr($content, $endOfStartTag + 1, $h1CloseIndex - $endOfStartTag -1));
      $h1Index = strpos($content, "<h1", $h1CloseIndex);
    }
    if(count($tabTitles) > 0)
    {
      array_push($tabContent, substr($content, $h1CloseIndex + 5));

      // generate the container
      $tabHTML = '<div id="tgei-auto-tabs-container">';
      $tabHTML .= '<div id="tgei-auto-tabs-tabs-container">';
      for($i = 0; $i < count($tabTitles); $i++)
      {
        $tabHTML .= '<button class="tgei-auto-tabs-tab" data-hash="#tab-'.$i.'">'.$tabTitles[$i].'</button>';
      }
      $tabHTML .= '
          </li>
        </div>
        <div id="tgei-auto-tabs-content-container">';
      for($i = count($tabContent) - 1; $i >= 0; $i--)
      {
        $tabHTML .= '<div class="tgei-auto-tabs-content" id="tab-'.$i.'">'.$tabContent[$i].'</div>';
      }
      $tabHTML .= '
          </div>
        </div>
      ';
      $contentMod .= $tabHTML;
      return $contentMod;
    }
    return $content; 
  }
}

TGEI_AutoTabs::register();
?>
